// utils
function slug(txt) {
    // Conserva caracteres de palabra Unicode, elimina el resto (simulando python's `re.sub(r"[^\w]", "", txt)`)
    return (txt || "").replace(/[^\p{L}\p{N}_]/gu, "").toLowerCase();
}

function is_url(t) {
    if (!t) return false;
    let lower = String(t).trim().toLowerCase();
    return lower.startsWith("http://") || lower.startsWith("https://");
}

function build_utm(url_str, camp, ident) {
    try {
        let u = new URL(url_str);
        // Si ya tiene utm_, devolvemos original
        let hasUtm = false;
        u.searchParams.forEach((v, k) => {
            if (k.startsWith("utm_")) hasUtm = true;
        });
        if (hasUtm) return url_str;

        u.searchParams.set("utm_source", "newsletter");
        u.searchParams.set("utm_medium", "email");
        u.searchParams.set("utm_campaign", camp);
        u.searchParams.set("utm_id", camp);
        u.searchParams.set("utm_content", ident);
        return u.toString();
    } catch (e) {
        return url_str;
    }
}

function getCellValue(ws, r, c) {
    let addr = XLSX.utils.encode_cell({ r: r, c: c });
    let cell = ws[addr];
    if (!cell) return { v: "", link: null };
    let val = cell.v !== undefined && cell.v !== null ? String(cell.v).trim() : "";
    let link = (cell.l && cell.l.Target) ? String(cell.l.Target).trim() : null;
    return { v: val, link: link };
}

function collect_campaigns(ws) {
    if (!ws['!ref']) return {};
    let range = XLSX.utils.decode_range(ws['!ref']);
    let R = range.e.r;
    let C = range.e.c;

    let etqs = [];
    for (let r = 0; r <= R; r++) {
        for (let c = 0; c <= C; c++) {
            let cell = getCellValue(ws, r, c);
            if (cell.v.toLowerCase() === "etiqueta") {
                etqs.push({ r: r, c: c });
            }
        }
    }

    if (etqs.length === 0) return {};
    // Ordenar etiquetas por columna, luego fila
    etqs.sort((a, b) => {
        if (a.c === b.c) return a.r - b.r;
        return a.c - b.c;
    });

    // Delimitador final (al igual que el código Python)
    etqs.push({ r: R + 1, c: 0 });

    let camps = {};
    for (let i = 0; i < etqs.length - 1; i++) {
        let et = etqs[i];

        let next_cols = etqs.filter(x => x.c > et.c).map(x => x.c);
        let col_end = next_cols.length > 0 ? Math.min(...next_cols) - 1 : C;

        let same_col = etqs.slice(i + 1).filter(x => x.c === et.c).map(x => x.r);
        let row_end = same_col.length > 0 ? Math.min(...same_col) - 1 : R;

        let camp = "";
        for (let cc = et.c + 1; cc <= col_end; cc++) {
            let val = getCellValue(ws, et.r, cc).v;
            if (val) {
                camp = val;
                break;
            }
        }
        if (!camp) continue;

        let rows = [];
        let seen = new Set();
        let last_label = "";

        for (let rr = et.r + 1; rr <= row_end; rr++) {
            for (let cc = et.c; cc <= col_end; cc++) {
                let val = getCellValue(ws, rr, cc).v;
                if (val && !is_url(val)) {
                    last_label = val;
                    break;
                }
            }
            let links = [];
            for (let cc = et.c; cc <= col_end; cc++) {
                let cellData = getCellValue(ws, rr, cc);
                let txt = cellData.v;
                let lk = (!txt && cellData.link) ? cellData.link : txt;
                if (lk && is_url(lk)) {
                    links.push(lk);
                }
            }
            if (links.length > 0 && last_label) {
                links.forEach(lk => {
                    let key = last_label + "|" + lk;
                    if (!seen.has(key)) {
                        seen.add(key);
                        rows.push({
                            label: last_label,
                            original: lk,
                            utm: build_utm(lk, camp, slug(last_label))
                        });
                    }
                });
            }
        }
        if (rows.length > 0) {
            camps[camp] = rows;
        }
    }
    return camps;
}

// UI State
let bookData = {};
let currentSheet = "";
let currentCountry = "";
let showUTM = true;

const COUNTRIES = {
    'mx': 'MÉXICO',
    'cl': 'CHILE',
    'ar': 'ARGENTINA',
    'co': 'COLOMBIA',
    'pa': 'PANAMÁ',
    'pe': 'PERÚ',
    'bz': 'BRASIL',
    'br': 'BRASIL',
    'es': 'ESPAÑA'
};

function extractCountry(campaignName) {
    const parts = campaignName.toLowerCase().split('_');
    const lastPart = parts[parts.length - 1];

    // Si la campaña termina explícitamente en el código de país
    if (COUNTRIES[lastPart]) {
        return lastPart.toLowerCase();
    }

    // Fallback: tratar de buscar en las partes finales
    for (let p of parts.slice(-3)) {
        if (COUNTRIES[p]) return p.toLowerCase();
    }

    return 'otros'; // Si no detecta país
}

function showView(viewId) {
    document.querySelectorAll('.view').forEach(v => v.classList.remove('active'));
    document.getElementById(viewId).classList.add('active');

    if (viewId === 'view-landing') {
        document.body.classList.add('is-landing');
    } else {
        document.body.classList.remove('is-landing');
    }
}

document.addEventListener('DOMContentLoaded', () => {
    // DOM Elements
    const fileInput = document.getElementById('fileInput');
    const btnProcess = document.getElementById('btnProcess');

    // File Processing
    btnProcess.addEventListener('click', () => {
        if (!fileInput.files.length) {
            alert("Por favor selecciona un archivo Excel (.xlsx, .xls)");
            return;
        }

        btnProcess.textContent = "Procesando...";
        btnProcess.disabled = true;

        let file = fileInput.files[0];
        let reader = new FileReader();
        reader.onload = function (e) {
            try {
                let data = new Uint8Array(e.target.result);
                // Leer archivo Excel usando SheetJS
                let workbook = XLSX.read(data, { type: 'array', cellHyperlinks: true });

                bookData = {};
                workbook.SheetNames.forEach(sh => {
                    let ws = workbook.Sheets[sh];
                    let campaigns = collect_campaigns(ws);
                    if (Object.keys(campaigns).length > 0) {
                        bookData[sh] = campaigns;
                    }
                });

                let hasData = Object.keys(bookData).length > 0;
                if (!hasData) {
                    alert("No se encontró ninguna campaña con enlaces y estructura de 'Etiquetas'.");
                    btnProcess.textContent = "Procesar archivo";
                    btnProcess.disabled = false;
                    return;
                }

                initResultsView();
            } catch (err) {
                console.error(err);
                alert("Error procesando Excel: " + err.message);
            } finally {
                btnProcess.textContent = "Procesar archivo";
                btnProcess.disabled = false;
            }
        };
        reader.onerror = function () {
            alert("Error leyendo el archivo.");
            btnProcess.textContent = "Procesar archivo";
            btnProcess.disabled = false;
        };
        reader.readAsArrayBuffer(file);
    });

    // Back to upload
    document.getElementById('btnBackToUpload').addEventListener('click', () => {
        showView('view-landing');
        fileInput.value = '';
    });

    // Back to results
    document.getElementById('btnBackToResults').addEventListener('click', () => {
        showView('view-results');
    });
});

function initResultsView() {
    const sheetSel = document.getElementById('sheetSel');
    sheetSel.innerHTML = '';

    const visibleSheets = Object.keys(bookData);

    if (visibleSheets.length === 0) {
        alert("No se encontraron hojas con campañas.");
        return;
    }

    visibleSheets.forEach(sh => sheetSel.add(new Option(sh, sh)));
    currentSheet = visibleSheets[0];
    sheetSel.value = currentSheet;

    sheetSel.onchange = e => {
        currentSheet = e.target.value;
        updateCountryTabs();
        renderCards();
    };

    const toggle = document.getElementById('toggleLinks');
    toggle.checked = true;
    showUTM = true;
    toggle.onchange = e => { showUTM = e.target.checked; renderCards(); };

    updateCountryTabs();
    renderCards();
    showView('view-results');
}

function updateCountryTabs() {
    const data = bookData[currentSheet] || {};
    const countries = new Set();

    Object.keys(data).forEach(campName => {
        countries.add(extractCountry(campName));
    });

    const countryArr = Array.from(countries).sort();
    const tabsContainer = document.getElementById('countryTabs');
    tabsContainer.innerHTML = '';

    if (countryArr.length === 0) return;

    if (!countryArr.includes(currentCountry)) {
        currentCountry = countryArr[0];
    }

    countryArr.forEach(c => {
        let label = COUNTRIES[c] || c.toUpperCase();
        if (c === 'otros') label = 'TODOS / OTROS';

        const btn = document.createElement('button');
        btn.className = 'country-tab' + (currentCountry === c ? ' active' : '');
        btn.textContent = label;
        btn.onclick = () => {
            currentCountry = c;
            updateCountryTabs();
            renderCards();
        };
        tabsContainer.appendChild(btn);
    });
}

function copy(txt, el) {
    if (el.classList.contains('selected-permanent')) {
        // Desmarcar si ya estaba en verde
        el.classList.remove('selected-permanent');
    } else {
        // Marcar y copiar al portapapeles
        navigator.clipboard.writeText(txt);
        el.classList.add('copied');
        el.classList.add('selected-permanent');
        setTimeout(() => el.classList.remove('copied'), 800);
    }
}

function renderCards() {
    const wrap = document.getElementById('sheetView');
    wrap.innerHTML = '';
    const data = bookData[currentSheet] || {};

    Object.entries(data).forEach(([name, rows]) => {
        const campCountry = extractCountry(name);

        // Filtrar por país si hay varios países
        if (currentCountry !== 'otros' && campCountry !== 'otros' && campCountry !== currentCountry) {
            return;
        }

        const card = document.createElement('div');
        card.className = 'card';

        let titleHtml = '';
        if (COUNTRIES[campCountry]) {
            titleHtml += `<span class="badge country">${COUNTRIES[campCountry]}</span>`;
        }
        titleHtml += `<span class="campaign-name" title="${name}">${name}</span>`;

        card.innerHTML = `
          <h2 title="${name}">${titleHtml}</h2>
          <div class="inner">
            <table>
              <thead><tr><th>Elemento</th><th>${showUTM ? 'UTM' : 'Original'}</th></tr></thead>
              <tbody></tbody>
            </table>
          </div>`;
        const tbody = card.querySelector('tbody');
        rows.forEach(r => {
            const tr = document.createElement('tr');
            tr.insertCell().textContent = r.label;
            const td = tr.insertCell();
            const display = showUTM ? r.utm : r.original;
            const toCopy = showUTM ? r.utm : r.original;

            const a = document.createElement('a');
            a.href = showUTM ? '#' : r.original;
            if (!showUTM) a.target = '_blank';
            a.className = 'url' + (showUTM ? ' copy' : '');
            a.textContent = display;

            if (showUTM) {
                a.onclick = e => { e.preventDefault(); copy(toCopy, a); };
            }
            td.appendChild(a);
            tbody.appendChild(tr);
        });
        wrap.appendChild(card);
    });
}


